<?php

namespace App\Http\Controllers;

use App\Models\PrevisionActivite;
use App\Models\Activite;
use App\Models\Extrant;
use Illuminate\Http\Request;

class PrevisionActivitesController extends Controller
{
	public function index() {
		
     try {
            $previsions = PrevisionActivite::with('activite', 'evaluations')
                ->orderBy('dateevaluation', 'asc')
                ->orderBy('activite_id', 'asc')
                ->orderBy('ordreevaluation', 'asc')
                ->get();

            return view('previsionactivites.index', compact('previsions'));
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Erreur lors du chargement des prévisions : ' . $e->getMessage());
        }
	}

	public function show($id) { 
	//	return PrevisionActivite::findOrFail($id); 
	 try {
				$prevision = PrevisionActivite::findOrFail($id);
				return $prevision;
			} catch (\Exception $e) {
				return redirect()->back()->with('error', 'Erreur lors de l’affichage de la prévision : ' . $e->getMessage());
			}
	  }
	
	public function create($activiteid, $extrantid) { 
		
     try {
            $canadd = Activite::where('id', $activiteid)->first();
			$extrant = Extrant::where('id', $extrantid)->first();

            if ($canadd->clos) {
                return redirect()->back()->with('error', 'L\'activité est déjà clôturée.');
            }

            if ($canadd->valide && $canadd->actif && !$canadd->ouvert) {
                return redirect()->back()->with('error', 'L\'activité est validée et fermée aux révisions.');
            }

            $lastPrevision = PrevisionActivite::where('activite_id', $activiteid)
                ->orderBy('ordreevaluation', 'desc')
                ->first();

            $nextOrdre = $lastPrevision ? $lastPrevision->ordreevaluation + 1 : 1;

            return view('previsionactivites.create', compact('activiteid', 'nextOrdre', 'canadd', 'extrant'));
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Erreur lors de l’ouverture du formulaire : ' . $e->getMessage());
        }
	}

    public function store(Request $request)
    {
		
       $data = $request->validate([
                'activite_id' => 'required|integer',
                'ordreevaluation' => 'required|numeric',
                'niveaucible' => 'required|numeric',
                'dateevaluation' => 'required|date',
				'extrant_id' => 'nullable|numeric',
            ]);
      try {
           

            $activite = Activite::findOrFail($data['activite_id']);
			
			//dd(isset($activite->extrant)); die();

            if ($activite->clos) {
                return redirect()->back()->with('error', 'L\'activité est déjà clôturée.');
            }

            if ($activite->valide && $activite->actif && !$activite->ouvert) {
                return redirect()->back()->with('error', 'L\'activité est validée et fermée aux prévisions.');
            }
			
			//S'assurer que la date ne sort pas de extrant
			if(isset($activite->extrant)&&($activite->extrant->debut > $data['dateevaluation'] || $activite->extrant->fin < $data['dateevaluation'])){
				return redirect()->back()->with('error', 'Date non compris dans l\'intervalle de l\'extrant (du '.extrant->debut.' au '.extrant->fin);
			}

            // Validation des cohérences
            $maxPrevision = PrevisionActivite::where('activite_id', $data['activite_id'])
				->where('extrant_id', $data['extrant_id'])
                ->where('dateevaluation', '<=', $data['dateevaluation'])
                ->orderBy('niveaucible', 'desc')
                ->orderBy('dateevaluation', 'desc')
                ->first();

            if ($maxPrevision && $data['niveaucible'] <= $maxPrevision->niveaucible) {
                return redirect()->back()->withInput()
                    ->with('error', 'La cible prévisionnelle ne peut pas être inférieure ou égale à celle du '
                        . $maxPrevision->dateevaluation . ' (' . $maxPrevision->niveaucible . '%).');
            }

            $minPrevision = PrevisionActivite::where('activite_id', $data['activite_id'])
				->where('extrant_id', $data['extrant_id'])
                ->where('dateevaluation', '>=', $data['dateevaluation'])
                ->orderBy('niveaucible', 'asc')
                ->orderBy('dateevaluation', 'asc')
                ->first();

            if ($minPrevision && $data['niveaucible'] >= $minPrevision->niveaucible) {
                return redirect()->back()->withInput()
                    ->with('error', 'La cible prévisionnelle ne peut pas être supérieure ou égale à celle du '
                        . $minPrevision->dateevaluation . ' (' . $minPrevision->niveaucible . '%).');
            }

            $prevision = PrevisionActivite::create($data);

            if ($prevision) {
                $activite->update(['actif' => false, 'ouvert' => false, 'valide' => false]);
                $this->renumeroterPrevisions($activite->id);
            }

            return redirect()->route('activites.planning', $activite->id)
                ->with('success', 'Prévision enregistrée avec succès.');
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Erreur lors de l’enregistrement de la prévision : ' . $e->getMessage());
        }
	}
	
	public function edit($id) { 
		
     try {
            $previsionactivite = PrevisionActivite::findOrFail($id);
            $activite = Activite::findOrFail($previsionactivite->activite_id);

            if ($activite->clos) {
                return redirect()->back()->with('error', 'L\'activité est déjà clôturée.');
            }

            if ($activite->valide && !$activite->ouvert) {
                return redirect()->back()->with('error', 'L\'activité est validée et fermée aux révisions.');
            }

            $maxbefore = PrevisionActivite::where('ordreevaluation', '<', $previsionactivite->ordreevaluation)
                ->orderBy('ordreevaluation', 'desc')
                ->first();

            $minafer = PrevisionActivite::where('ordreevaluation', '>', $previsionactivite->ordreevaluation)
                ->orderBy('ordreevaluation', 'asc')
                ->first();

            return view('previsionactivites.edit', compact('previsionactivite', 'maxbefore', 'minafer', 'activite'));
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Erreur lors de l’ouverture du formulaire d’édition : ' . $e->getMessage());
        }
	}

    public function update(Request $request, $id)
    {
		
       $data = $request->validate([
                'activite_id' => 'required|integer',
                'ordreevaluation' => 'required|numeric',
                'dateevaluation' => 'required|date',
                'niveaucible' => 'required|numeric',
				'extrant_id' => 'nullable|numeric',
            ]);
      
      try {
           

            $activite = Activite::findOrFail($data['activite_id']);

            if ($activite->clos) {
                return redirect()->back()->with('error', 'L\'activité est déjà clôturée.');
            }

            // Vérifications cohérentes
            $maxPrevision = PrevisionActivite::where('activite_id', $data['activite_id'])
				->where('extrant_id', $data['extrant_id'])
                ->where('id', '!=', $id)
                ->where('dateevaluation', '<=', $data['dateevaluation'])
                ->orderBy('niveaucible', 'desc')
                ->first();

            if ($maxPrevision && $data['niveaucible'] <= $maxPrevision->niveaucible) {
                return redirect()->back()->withInput()
                    ->with('error', 'La cible prévisionnelle ne peut pas être inférieure ou égale à celle du '
                        . $maxPrevision->dateevaluation . ' (' . $maxPrevision->niveaucible . '%).');
            }

            $minPrevision = PrevisionActivite::where('activite_id', $data['activite_id'])
				->where('extrant_id', $data['extrant_id'])
                ->where('id', '!=', $id)
                ->where('dateevaluation', '>=', $data['dateevaluation'])
                ->orderBy('niveaucible', 'asc')
                ->first();

            if ($minPrevision && $data['niveaucible'] >= $minPrevision->niveaucible) {
                return redirect()->back()->withInput()
                    ->with('error', 'La cible prévisionnelle ne peut pas être supérieure ou égale à celle du '
                        . $minPrevision->dateevaluation . ' (' . $minPrevision->niveaucible . '%).');
            }

            $prevision = PrevisionActivite::findOrFail($id);
            $ok = $prevision->update($data);
            $this->renumeroterPrevisions($prevision->activite_id);

            if ($ok) {
                $activite->update(['actif' => false, 'ouvert' => false, 'valide' => false]);
            }

            return redirect()->route('activites.planning', $activite->id)
                ->with('success', 'Prévision mise à jour avec succès.');
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Erreur lors de la mise à jour : ' . $e->getMessage());
        }
    }

    public function destroy($id)
	{
		// Récupérer la prévision à supprimer
		$prevision = PrevisionActivite::findOrFail($id);
		$activite_id = $prevision->activite_id;
		$activite = Activite::findOrFail($activite_id);
		$extrant_id = $prevision->extrant_id;
		
		$cant_delete = $prevision->evaluations()->exists();
			//dd($cant_delete); die();
		if($cant_delete) {
			return redirect()->back()->with('error', 'Impossible. Prévision liéé à une évaluation');
		}
		
		if($activite->clos){
			return redirect()->back()->with('error', 'L\'activité est déjà cloturée.');
		}
		
		// Supprimer la prévision
		$prevision->delete();

		// Récupérer toutes les prévisions restantes pour cette activité et extrant, triées par ordre
		$previsions = PrevisionActivite::where('activite_id', $activite_id)
						->where('extrant_id', $extrant_id)
						->orderBy('ordreevaluation', 'asc')
						->get();

		// Réorganiser les ordres (1, 2, 3, ...)
		$ordre = 1;
		foreach ($previsions as $p) {
			$p->ordreevaluation = $ordre++;
			$p->save();
		}

		return redirect()->back()->with('success', 'Étape supprimée et prévisions réorganisées.');
	}

	
	public function renumeroterPrevisions($activite_id)
	{
		// Récupérer toutes les prévisions de l'activité, triées par date croissante
		$previsions = PrevisionActivite::where('activite_id', $activite_id)
			->orderBy('extrant_id', 'asc')
			->orderBy('dateevaluation', 'asc')
			->get();

		// Parcourir et réaffecter les ordres
		$currentExtrantId = null;
		$ordre = 1;
		foreach ($previsions as $prevision) {
			
			// Si on change d'extrant, réinitialiser l'ordre
			if ($prevision->extrant_id !== $currentExtrantId) {
				$currentExtrantId = $prevision->extrant_id;
				$ordre = 1;
			}
			// Mettre à jour seulement si l'ordre a changé
			if ($prevision->ordreevaluation != $ordre) {
				$prevision->ordreevaluation = $ordre;
				$prevision->save();
			}
			$ordre++;
		}
	}
}
