/**
 * Exporte un tableau en Excel avec possibilité de sélectionner les colonnes
 * @param {string} tableId - ID du tableau HTML
 * @param {string} filename - Nom du fichier à télécharger
 * @param {array} excludeColumns - Indices des colonnes à exclure (ex: [0, 3])
 * @param {array} includeColumns - Indices des colonnes à inclure (si null, prend toutes sauf excludeColumns)
 */
function exportTableToExcel(tableId, filename = 'export.xlsx', options = {}) {
    const {
        excludeColumns = [],
        includeColumns = null,
        sheetName = 'Sheet1'
    } = options;
    
    const table = document.getElementById(tableId);
    if (!table) {
        console.error('Tableau non trouvé');
        return;
    }
    
    // Clone le tableau pour ne pas modifier l'original
    const clonedTable = table.cloneNode(true);
    
    // Supprime les colonnes non désirées
    const rows = clonedTable.querySelectorAll('tr');
    rows.forEach(row => {
        const cells = Array.from(row.children);
        
        // Si includeColumns est défini, on garde seulement ces colonnes
        if (includeColumns && includeColumns.length > 0) {
            cells.forEach((cell, index) => {
                if (!includeColumns.includes(index)) {
                    cell.remove();
                }
            });
        } 
        // Sinon on exclut les colonnes spécifiées
        else if (excludeColumns.length > 0) {
            // On parcourt en ordre inversé pour éviter les problèmes d'index
            for (let i = cells.length - 1; i >= 0; i--) {
                if (excludeColumns.includes(i)) {
                    cells[i].remove();
                }
            }
        }
    });
    
    // Crée le workbook
    const wb = XLSX.utils.table_to_book(clonedTable, {sheet: sheetName});
    
    // Télécharge le fichier
    XLSX.writeFile(wb, filename);
}