<?php

namespace App\Http\Controllers;

use App\Models\Extrant;
use App\Models\Activite;
use Illuminate\Http\Request;
use Carbon\Carbon;

class ExtrantsController extends Controller
{
    public function index() { 
	return Extrant::all(); }
	
	public function create($activiteid)
	{
		// On peut vérifier que l'activité existe (optionnel)
		// $activite = Activite::findOrFail($activiteid);

		return view('extrants.create', [
			'activite_id' => $activiteid
		]);
	}

	public function store(Request $request)
	{
		$validated = $request->validate([
			'nom' => 'required|string|max:255',
			'activite_id' => 'required|integer|exists:activites,id',
			'annee' => 'required|integer|min:2000|max:2100',
			// on ignore 'fin', elle sera recalculée
		]);
		
		 // Vérifier si un extrant existe déjà pour cette activité et année
		$exists = Extrant::where('activite_id', $validated['activite_id'])
						  ->whereYear('debut', $validated['annee'])
						  ->exists();

		if ($exists) {
			return redirect()->back()
							 ->withInput()
							 ->withErrors(['activite_id' => 'Un extrant existe déjà pour cette activité et cette année.']);
		}

		// Calculer début et fin de l'année
		$debut = Carbon::createFromDate($validated['annee'], 1, 1)->toDateString();
		$fin = Carbon::createFromDate($validated['annee'], 12, 31)->toDateString();


		// Créer l'extrant
		$ok = $extrant = Extrant::create([
			'nom' => $validated['nom'],
			'activite_id' => $validated['activite_id'],
			'debut' => $debut,
			'fin' => $fin,
		]);
		
		if ($ok) {
			$activite = Activite::findOrFail($validated['activite_id']);
			$activite->update(['actif' => false, 'ouvert' => false, 'valide' => false]);
        }

		return redirect()
        ->route('activites.show', $extrant->activite_id)
        ->with('success', 'Extrant créé avec succès pour l’année ' . $validated['annee'] . '.');
	}
	
	public function edit($id)
    {
        $extrant = Extrant::findOrFail($id);
		if ($extrant->previsions()->exists() || $extrant->evaluations()->exists()) {
			return redirect()->back()->with('error', 'Erreur de suppression. Des évaluations ou prévisions existent pour cet extrant ');
		}

        return view('extrants.edit', compact('extrant'));
    }

    public function update(Request $request, $id)
    {
        $extrant = Extrant::findOrFail($id);

        $validated = $request->validate([
            'nom' => 'required|string|max:255',
            'annee' => 'required|integer|min:2000|max:2100',
        ]);
		
		// Vérifier si un autre extrant existe pour la même activité et la même année
		$exists = Extrant::where('activite_id', $extrant->activite_id)
						 ->whereYear('debut', $validated['annee'])
						 ->where('id', '!=', $extrant->id)
						 ->exists();

		if ($exists) {
			return redirect()->back()
							 ->withInput()
							 ->withErrors(['annee' => 'Un extrant existe déjà pour cette activité et cette année.']);
		}

        // Calculer début et fin de l'année
		$debut = Carbon::createFromDate($validated['annee'], 1, 1)->toDateString();
		$fin = Carbon::createFromDate($validated['annee'], 12, 31)->toDateString();

        // Mettre à jour l'extrant
		$ok = $extrant->update([
			'nom' => $validated['nom'],
			'debut' => $debut,
			'fin' => $fin,
		]);
		
		
		if ($ok) {
			$activite = Activite::findOrFail($extrant->activite_id);
			$activite->update(['actif' => false, 'ouvert' => false, 'valide' => false]);
        }

		return redirect()
			->route('activites.show', $extrant->activite_id)
			->with('success', 'Extrant mis à jour avec succès pour l’année ' . $validated['annee'] . '.');
    }
	
	public function destroy($id)
    {
        $extrant = Extrant::findOrFail($id);
        $activiteId = $extrant->activite_id;
		
		if ($extrant->previsions()->exists() || $extrant->evaluations()->exists()) {
			return redirect()->back()->with('error', 'Erreur de suppression. Des évaluations ou prévisions existent pour cet extrant ');
		}

        $extrant->delete();

        return redirect()
            ->route('activites.show', $activiteId)
            ->with('success', 'Extrant supprimé avec succès.');
    }
}
